// $Id: CDelayPanel.cpp,v 1.5 2007/02/11 02:20:27 paul Exp $

/*
 * All contents of this source code are copyright 2007 Exp Digital Uk.
 * This file forms part of the example projects for the Infinity API.
 * You may not redistribute either the source code or the compiled binary.
 * The Infinity API is covered by a fair use licence which you should have recieved with this
 * file. If you didnt receieve the licence, please contact us via http://www.expdigital.co.uk
 */

#include "CDelayPanel.hpp"
#include "CSCalWindow.hpp"
using Exponent::SCal::CSCalWindow;
using Exponent::SCal::CDelayPanel;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CDelayPanel, CControlPanel);

//	===========================================================================
CDelayPanel::CDelayPanel(IControlRoot *root, const long uniqueId, const CRect &area, CMenu *divisionMenu)
		   : CControlPanel(root->getParentWindow(), root, uniqueId, area, NULL)
		   , m_delayBpmEditBox(NULL)
		   , m_delayDivisionComboBox(NULL)
		   , m_delayOutputValueLabel(NULL)
		   , m_delayDivision(CMusicMath::e_quarterBar)
{
	EXPONENT_CLASS_CONSTRUCTION(CDelayPanel);

	this->onDrawDrawThePanelBounds(false);

	// BPM edit
	m_delayBpmEditBox		  = new CTextEdit(this,  e_delayBpmEditBox,				    CSCalWindow::CSCALWINDOW_CONTROL1_RECT, "120.00", this);
	m_delayDivisionComboBox	  = new CComboBox(this,  e_delayDivisionComboBox,			CSCalWindow::CSCALWINDOW_CONTROL2_RECT, "1/4", divisionMenu, this);
	m_delayOutputValueLabel	  = new CTextLabel(this, CControl::CCONTROL_NO_ID_REQUIRED, CSCalWindow::CSCALWINDOW_OUTPUTBOX_RECT, CString::CSTRING_EMPTY_STRING);

	// Reference
	m_delayBpmEditBox->referenced();
	m_delayDivisionComboBox->referenced();
	m_delayOutputValueLabel->referenced();

	// Set the format
	m_delayBpmEditBox->setTextFormat(CGraphics::e_centerCenterAlign);
	m_delayDivisionComboBox->setTextFormat(CGraphics::e_centerCenterAlign);
	m_delayOutputValueLabel->setTextFormat(CGraphics::e_centerCenterAlign);
	m_delayBpmEditBox->setAlignment(CTextEdit::e_center);

	// Make single click editable
	m_delayBpmEditBox->setSingleClickEditable();

	// Colours
	m_delayBpmEditBox->setColours(			CAlphaColour::CALPHACOLOUR_BLACK, CSCalWindow::CSCALWINDOW_BACKGROUND_COLOUR, CSCalWindow::CSCALWINDOW_TEXT_COLOUR);
	m_delayDivisionComboBox->setColours(	CAlphaColour::CALPHACOLOUR_BLACK, CSCalWindow::CSCALWINDOW_BACKGROUND_COLOUR, CSCalWindow::CSCALWINDOW_TEXT_COLOUR);
	m_delayOutputValueLabel->setColours(	CAlphaColour::CALPHACOLOUR_BLACK, CAlphaColour::CALPHACOLOUR_WHITE, CSCalWindow::CSCALWINDOW_ANSWER_TEXT_COLOUR);

	// Create the attributes to pass to the windows
	m_delayDivisionComboBox->setMenuWindowAttributes(&CSCalWindow::CSCALWINDOW_MENU_WINDOW_ATTRIBUTES, false);
	m_delayDivisionComboBox->setFont(CFont::CFONT_SYSTEM_FONT);

	// Colours of menus. Notice that this *must* be called after the menu window attributes, otherwise there isnt a window to open
	m_delayDivisionComboBox->setMenuColours(CSCalWindow::CSCALWINDOW_MENU_COLOURS);

	// Add controls
	this->addControl(m_delayBpmEditBox);
	this->addControl(m_delayDivisionComboBox);
	this->addControl(m_delayOutputValueLabel);

	// Compute
	this->computeDelayTime();
}

//	===========================================================================
CDelayPanel::~CDelayPanel()
{
	EXPONENT_CLASS_DESTRUCTION(CDelayPanel);
	FORGET_COUNTED_OBJECT(m_delayBpmEditBox);
	FORGET_COUNTED_OBJECT(m_delayDivisionComboBox);
	FORGET_COUNTED_OBJECT(m_delayOutputValueLabel);
}

//	===========================================================================
void CDelayPanel::handleActionEvent(const CActionEvent &event)
{
	switch(event.getControl()->getUniqueId())
	{
		case e_delayDivisionComboBox:
			{
				const CMenuItem *item = m_delayDivisionComboBox->getLastMenuItem();

				// Check item is valid
				if (item == NULL)
				{
					CDialog::notifyUser("Please select a valid tempo division", "SCal Error");
					return;
				}

				// Store the id
				m_delayDivision = (CMusicMath::ETempoDivision)item->getId();
			}
			// Fall through
		case e_delayBpmEditBox:
			this->computeDelayTime();
		break;
	}
}

//	===========================================================================
void CDelayPanel::computeDelayTime()
{
	double bpm = CString::toDouble(m_delayBpmEditBox->getString().getString());
	
	if (bpm <= 0.0 || bpm >= 1000.0)
	{
		CDialog::notifyUser("Bpm range error!\n\n0.0 < bpm < 1000.0\n\nBpm must be between 0 and 1000 bpm", "SCal error", true);
		bpm = 120.0;
		m_delayBpmEditBox->setText("120.00");
	}
	else
	{
		CString string;
		string.setStringWithFormat("%0.02f", bpm);
		m_delayBpmEditBox->setText(string);
	}
	m_delayBpmEditBox->update();

	// Store the output
	CString output;
	output.setStringWithFormat("%0.02f", CMusicMath::getRateInMilliseconds(bpm, m_delayDivision, CTimeSignature(4, 4)));
	m_delayOutputValueLabel->setText(output);
	m_delayOutputValueLabel->update();
}